//////////////////////////////////////////////////////////////////
//                                                              //
//  METRICSMONITOR SERVER SCRIPT FOR FM-DX-WEBSERVER  (V1.5B5)  //
//                                                              //
//  by Highpoint               last update: 01.01.2026          //
//                                                              //
//  Thanks for support by                                       //
//  Jeroen Platenkamp, Bkram, Wötkylä, AmateurAudioDude         //
//                                                              //
//  https://github.com/Highpoint2000/metricsmonitor             //
//                                                              //
//////////////////////////////////////////////////////////////////

// ====================================================================================
//  DEBUG CONFIGURATION
//  Set to 'true' to enable detailed logging of MPX/RDS/SNR values to the console.
//  This is useful for calibrating the input levels or debugging signal issues.
// ====================================================================================
const ENABLE_EXTENDED_LOGGING = false;

// ====================================================================================
//  MODULE IMPORTS
//  We need these built-in Node.js modules and external dependencies to function.
// ====================================================================================
const { spawn, execSync } = require("child_process");
const WebSocket = require("ws");
const fs = require("fs");
const path = require("path");

// Import core server utilities for logging and configuration
// These paths assume the standard file structure of the FM-DX-Webserver
const { logInfo, logError, logWarn } = require("./../../server/console");
const mainConfig = require("./../../config.json");

// ====================================================================================
//  PLUGIN CONFIGURATION MANAGEMENT
//  Handles loading, validating, and normalizing the 'metricsmonitor.json' config file.
// ====================================================================================

// Path to the configuration file
const configFilePath = path.join(
  __dirname,
  "./../../plugins_configs/metricsmonitor.json"
);

/**
 * DEFAULT CONFIGURATION OBJECT
 * These values are used if the config file is missing or specific keys are undefined.
 * The order here reflects the user's requested JSON structure.
 */
const defaultConfig = {
  // 1. Audio & MPX Hardware Settings
  sampleRate: 192000,           // The sample rate for capture (Hz)
  MPXmode: "auto",              // Mode switch (off/auto/on)
  MPXStereoDecoder: "off",      // Internal stereo decoder switch
  MPXInputCard: "",             // Input device name (if empty, tries to use 3LAS)

  // 2. Calibration Offsets (Meters)
  MeterInputCalibration: 0.0,   // Input Gain Calibration in dB (applies to MPX METERS only)
  MeterPilotCalibration: 0.0,	// Pilot Calibration in dB
  MeterMPXCalibration: 0.0,     // MPX Calibration in dB
  MeterRDSCalibration: 0.0,     // RDS Calibration in dB

  // 3. FFT / Spectrum Settings
  fftLibrary: "pffft.wasm",     // "fft-js" or "pffft.wasm" (faster)
  fftSize: 4096,                // FFT Window size (resolution)
  
  // 4. Spectrum Visuals
  SpectrumInputCalibration: 0,  // Input Gain Calibration in dB (applies to SPECTRUM only)
  SpectrumAttackLevel: 3,       // Smoothing attack
  SpectrumDecayLevel: 15,       // Smoothing decay
  SpectrumSendInterval: 30,     // WebSocket update rate (approx 30fps)
  "Spectrum-Y-Offset": -40,     // Y-Axis offset for the visual curve
  "Spectrum-Y-Dynamics": 2,     // Dynamic range scaling for the visual curve

  // 5. Meter Gains
  StereoBoost: 2.3,             // Multiplier for L/R stereo meters
  AudioMeterBoost: 1.0,         // Multiplier for 5-Band audiometer

  // 6. Layout & UI
  MODULE_SEQUENCE: "1,2,0,3,4", // Order of UI modules
  CANVAS_SEQUENCE: "2,4",       // Order of Canvas elements
  LockVolumeSlider: true,       // Lock the main volume slider in UI
  EnableSpectrumOnLoad: true,   // Start spectrum automatically

  // 7. Colors & Peaks
  MeterColorSafe: "rgb(0, 255, 0)",     // RGB Array (Green)
  MeterColorWarning: "rgb(255, 255,0)", // RGB Array (Yellow)
  MeterColorDanger: "rgb(255, 0, 0)",   // RGB Array (Red)
  PeakMode: "dynamic",                  // "dynamic" or "fixed"
  PeakColorFixed: "rgb(251, 174, 38)"   // RGB Color for fixed peak
};

/**
 * NORMALIZE PLUGIN CONFIGURATION
 * Ensures that the loaded JSON object contains all necessary keys.
 * Migrates old/deprecated keys to new names if found.
 */
function normalizePluginConfig(json) {
  // 1. Migration: Rename minSendIntervalMs -> SpectrumSendInterval
  if (typeof json.minSendIntervalMs !== "undefined" && typeof json.SpectrumSendInterval === "undefined") {
    json.SpectrumSendInterval = json.minSendIntervalMs;
    delete json.minSendIntervalMs;
  }
  
  // 2. Migration: Rename Curve-Y-Offset -> Spectrum-Y-Offset
  if (typeof json["Curve-Y-Offset"] !== "undefined" && typeof json["Spectrum-Y-Offset"] === "undefined") {
    json["Spectrum-Y-Offset"] = json["Curve-Y-Offset"];
    delete json["Curve-Y-Offset"];
  }

  // 3. Migration: Rename Curve-Y-Dynamics -> Spectrum-Y-Dynamics
  if (typeof json["Curve-Y-Dynamics"] !== "undefined" && typeof json["Spectrum-Y-Dynamics"] === "undefined") {
    json["Spectrum-Y-Dynamics"] = json["Curve-Y-Dynamics"];
    delete json["Curve-Y-Dynamics"];
  }

  // 4. Migration: Rename stereoBoost -> StereoBoost
  if (typeof json.stereoBoost !== "undefined" && typeof json.StereoBoost === "undefined") {
    json.StereoBoost = json.stereoBoost;
    delete json.stereoBoost;
  }

  // 5. Migration: Rename eqBoost / EqBoost -> AudioMeterBoost
  if (typeof json.AudioMeterBoost === "undefined") {
    if (typeof json.eqBoost !== "undefined") {
      json.AudioMeterBoost = json.eqBoost;
      delete json.eqBoost;
    } else if (typeof json.EqBoost !== "undefined") {
      json.AudioMeterBoost = json.EqBoost;
      delete json.EqBoost;
    }
  }

  // 6. Migration: ExtStereoDecoder -> MPXStereoDecoder
  if (typeof json.ExtStereoDecoder !== "undefined" && typeof json.MPXStereoDecoder === "undefined") {
    json.MPXStereoDecoder = json.ExtStereoDecoder;
    delete json.ExtStereoDecoder; 
  }

  // 7. Migration: MPXinputCalibration/MPXboost -> MeterInputCalibration
  if (typeof json.MeterInputCalibration === "undefined") {
    if (typeof json.MPXinputCalibration !== "undefined") {
        json.MeterInputCalibration = json.MPXinputCalibration;
        delete json.MPXinputCalibration;
    } else if (typeof json.MPXboost !== "undefined") {
        json.MeterInputCalibration = json.MPXboost;
        delete json.MPXboost;
    }
  }

  // 8. Migration: pilotCalibration -> MeterPilotCalibration
  if (typeof json.pilotCalibration !== "undefined" && typeof json.MeterPilotCalibration === "undefined") {
    json.MeterPilotCalibration = json.pilotCalibration;
    delete json.pilotCalibration;
  }

  // 9. Migration: mpxCalibration -> MeterMPXCalibration
  if (typeof json.mpxCalibration !== "undefined" && typeof json.MeterMPXCalibration === "undefined") {
    json.MeterMPXCalibration = json.mpxCalibration;
    delete json.mpxCalibration;
  }

  // 10. Migration: rdsCalibration -> MeterRDSCalibration
  if (typeof json.rdsCalibration !== "undefined" && typeof json.MeterRDSCalibration === "undefined") {
    json.MeterRDSCalibration = json.rdsCalibration;
    delete json.rdsCalibration;
  }
  
  // 11. Migration: CurveInputCalibration -> SpectrumInputCalibration
  if (typeof json.CurveInputCalibration !== "undefined" && typeof json.SpectrumInputCalibration === "undefined") {
    json.SpectrumInputCalibration = json.CurveInputCalibration;
    delete json.CurveInputCalibration;
  }

  // Cleanup: Remove unused keys
  if (typeof json.SpectrumAverageLevel !== "undefined") delete json.SpectrumAverageLevel;
  if (typeof json.DevLimitKHz !== "undefined") delete json.DevLimitKHz;
  if (typeof json.DevRefKHz !== "undefined") delete json.DevRefKHz;
  if (typeof json.DevUncKHz !== "undefined") delete json.DevUncKHz;
  if (typeof json.DevScaleKHzPerAmp !== "undefined") delete json.DevScaleKHzPerAmp;

  // Apply Defaults for missing keys
  const result = {
    sampleRate: typeof json.sampleRate !== "undefined" ? json.sampleRate : defaultConfig.sampleRate,
    MPXmode: typeof json.MPXmode !== "undefined" ? json.MPXmode : defaultConfig.MPXmode,
    MPXStereoDecoder: typeof json.MPXStereoDecoder !== "undefined" ? json.MPXStereoDecoder : defaultConfig.MPXStereoDecoder,
    MPXInputCard: typeof json.MPXInputCard !== "undefined" ? json.MPXInputCard : defaultConfig.MPXInputCard,
    
    MeterInputCalibration: typeof json.MeterInputCalibration !== "undefined" ? json.MeterInputCalibration : defaultConfig.MeterInputCalibration,
    MeterPilotCalibration: typeof json.MeterPilotCalibration !== "undefined" ? json.MeterPilotCalibration : defaultConfig.MeterPilotCalibration,
    MeterMPXCalibration: typeof json.MeterMPXCalibration !== "undefined" ? json.MeterMPXCalibration : defaultConfig.MeterMPXCalibration,
    MeterRDSCalibration: typeof json.MeterRDSCalibration !== "undefined" ? json.MeterRDSCalibration : defaultConfig.MeterRDSCalibration,
    
    fftLibrary: typeof json.fftLibrary !== "undefined" ? json.fftLibrary : defaultConfig.fftLibrary,
    fftSize: typeof json.fftSize !== "undefined" ? json.fftSize : defaultConfig.fftSize,
    
    SpectrumInputCalibration: typeof json.SpectrumInputCalibration !== "undefined" ? json.SpectrumInputCalibration : defaultConfig.SpectrumInputCalibration,
    SpectrumAttackLevel: typeof json.SpectrumAttackLevel !== "undefined" ? json.SpectrumAttackLevel : defaultConfig.SpectrumAttackLevel,
    SpectrumDecayLevel: typeof json.SpectrumDecayLevel !== "undefined" ? json.SpectrumDecayLevel : defaultConfig.SpectrumDecayLevel,
    SpectrumSendInterval: typeof json.SpectrumSendInterval !== "undefined" ? json.SpectrumSendInterval : defaultConfig.SpectrumSendInterval,
    "Spectrum-Y-Offset": typeof json["Spectrum-Y-Offset"] !== "undefined" ? json["Spectrum-Y-Offset"] : defaultConfig["Spectrum-Y-Offset"],
    "Spectrum-Y-Dynamics": typeof json["Spectrum-Y-Dynamics"] !== "undefined" ? json["Spectrum-Y-Dynamics"] : defaultConfig["Spectrum-Y-Dynamics"],
    
    StereoBoost: typeof json.StereoBoost !== "undefined" ? json.StereoBoost : defaultConfig.StereoBoost,
    AudioMeterBoost: typeof json.AudioMeterBoost !== "undefined" ? json.AudioMeterBoost : defaultConfig.AudioMeterBoost,
    
    MODULE_SEQUENCE: typeof json.MODULE_SEQUENCE !== "undefined" ? json.MODULE_SEQUENCE : defaultConfig.MODULE_SEQUENCE,
    CANVAS_SEQUENCE: typeof json.CANVAS_SEQUENCE !== "undefined" ? json.CANVAS_SEQUENCE : defaultConfig.CANVAS_SEQUENCE,
    LockVolumeSlider: typeof json.LockVolumeSlider !== "undefined" ? json.LockVolumeSlider : defaultConfig.LockVolumeSlider,
    EnableSpectrumOnLoad: typeof json.EnableSpectrumOnLoad !== "undefined" ? json.EnableSpectrumOnLoad : defaultConfig.EnableSpectrumOnLoad,
    
    MeterColorSafe: typeof json.MeterColorSafe !== "undefined" ? json.MeterColorSafe : defaultConfig.MeterColorSafe,
    MeterColorWarning: typeof json.MeterColorWarning !== "undefined" ? json.MeterColorWarning : defaultConfig.MeterColorWarning,
    MeterColorDanger: typeof json.MeterColorDanger !== "undefined" ? json.MeterColorDanger : defaultConfig.MeterColorDanger,
    PeakMode: typeof json.PeakMode !== "undefined" ? json.PeakMode : defaultConfig.PeakMode,
    PeakColorFixed: typeof json.PeakColorFixed !== "undefined" ? json.PeakColorFixed : defaultConfig.PeakColorFixed,
  };

  // Preserve any extra custom keys
  for (const key of Object.keys(json)) {
    if (!(key in result)) {
      result[key] = json[key];
    }
  }

  return result;
}

/**
 * LOAD OR CREATE CONFIG FILE
 * Reads the JSON file. If missing or corrupt, creates a new one with defaults.
 * Also handles creating a .bak backup before overwriting.
 * 
 * @param {string} filePath - Absolute path to the config file
 * @returns {Object} - The usable configuration object
 */
function loadConfig(filePath) {
  const dir = path.dirname(filePath);

  // Ensure directory exists
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir, { recursive: true });
  }

  if (fs.existsSync(filePath)) {
    try {
      const raw = fs.readFileSync(filePath, "utf8").trim();

      if (raw.length === 0) {
        throw new Error("Empty JSON file");
      }

      let json = JSON.parse(raw);

      if (!json || Object.keys(json).length === 0) {
        throw new Error("Empty JSON object");
      }

      // Normalize
      json = normalizePluginConfig(json);
      
      // CREATE BACKUP BEFORE OVERWRITING
      try {
        const backupPath = filePath + ".bak";
        fs.copyFileSync(filePath, backupPath);
      } catch (backupErr) {
        logWarn(`[MPX] Failed to create config backup: ${backupErr.message}`);
      }

      // Write back with new order/keys
      fs.writeFileSync(filePath, JSON.stringify(json, null, 2), "utf8");

      return json;
    } catch (err) {
      logWarn(
        "[MPX] metricsmonitor.json invalid → rewriting with defaults:",
        err.message
      );
      // Backup defaults
      fs.writeFileSync(
        filePath,
        JSON.stringify(defaultConfig, null, 2),
        "utf8"
      );
      return defaultConfig;
    }
  }

  // File does not exist, create it
  logWarn(
    "[MPX] metricsmonitor.json not found → creating new file with defaults."
  );
  fs.writeFileSync(filePath, JSON.stringify(defaultConfig, null, 2), "utf8");
  return defaultConfig;
}

// Load the configuration now
const configPlugin = loadConfig(configFilePath);

// ====================================================================================
//  CONFIGURATION VALUE EXTRACTION
//  Extract values into const/let variables for cleaner usage in the script.
// ====================================================================================

// Sequences can be arrays or comma-separated strings
let MODULE_SEQUENCE = configPlugin.MODULE_SEQUENCE;
let CANVAS_SEQUENCE = configPlugin.CANVAS_SEQUENCE;

// Sample Rate
const ANALYZER_SAMPLE_RATE = Number(configPlugin.sampleRate) || 192000;
const CONFIG_SAMPLE_RATE = ANALYZER_SAMPLE_RATE;

// Audio processing parameters
const STEREO_BOOST = Number(configPlugin.StereoBoost) || 1.0;
const AUDIO_METER_BOOST = Number(configPlugin.AudioMeterBoost) || 1.0;
const FFT_LIBRARY = String(configPlugin.fftLibrary).trim();
const FFT_SIZE = Number(configPlugin.fftSize) || 4096;
const SPECTRUM_SEND_INTERVAL = Number(configPlugin.SpectrumSendInterval) || 30;

// ====================================================================================
//  CALIBRATION HANDLING (dB to Linear)
// ====================================================================================

// 1. Meters Calibration (MeterInputCalibration)
const METER_INPUT_CALIBRATION_DB = Number(configPlugin.MeterInputCalibration) || 0;
// Convert dB to Linear Gain Factor: Gain = 10^(dB/20)
const METER_GAIN_FACTOR = Math.pow(10, METER_INPUT_CALIBRATION_DB / 20.0);

// 2. Spectrum Calibration (SpectrumInputCalibration)
const SPECTRUM_INPUT_CALIBRATION_DB = Number(configPlugin.SpectrumInputCalibration) || 0;
// Convert dB to Linear Gain Factor for Spectrum
const SPECTRUM_GAIN_FACTOR = Math.pow(10, SPECTRUM_INPUT_CALIBRATION_DB / 20.0);


// Deviation / ITU settings (Hardcoded)
const DEV_LIMIT_KHZ = 75;
const DEV_REF_KHZ = 19;
const DEV_UNC_KHZ = 2;

// Visual settings
const SPECTRUM_ATTACK_LEVEL = Number(configPlugin.SpectrumAttackLevel) || 3;
const SPECTRUM_DECAY_LEVEL = Number(configPlugin.SpectrumDecayLevel) || 15;
const MPX_MODE = String(configPlugin.MPXmode || "auto").toLowerCase();
const MPX_STEREO_DECODER = String(configPlugin.MPXStereoDecoder || "off").toLowerCase();
const MPX_INPUT_CARD = String(configPlugin.MPXInputCard || "").replace(/^["'](.*)["']$/, "$1");
const LOCK_VOLUME_SLIDER = configPlugin.LockVolumeSlider === true;
const ENABLE_SPECTRUM_ON_LOAD = configPlugin.EnableSpectrumOnLoad === true;

// Calibrations
const METER_PILOT_CALIBRATION = Number(configPlugin.MeterPilotCalibration) || 0.0;
const METER_MPX_CALIBRATION = Number(configPlugin.MeterMPXCalibration) || 0.0;
const METER_RDS_CALIBRATION = Number(configPlugin.MeterRDSCalibration) || 0.0;

// Curve adjustments
const SPECTRUM_Y_OFFSET = Number(configPlugin["Spectrum-Y-Offset"]) || -40;
const SPECTRUM_Y_DYNAMICS = Number(configPlugin["Spectrum-Y-Dynamics"]) || 2.0;

// Color & Peak Settings
const METER_COLOR_SAFE = JSON.stringify(configPlugin.MeterColorSafe || "rgb(0, 255, 0)");
const METER_COLOR_WARNING = JSON.stringify(configPlugin.MeterColorWarning || "rgb(255, 255, 0)");
const METER_COLOR_DANGER = JSON.stringify(configPlugin.MeterColorDanger || "rgb(255, 0, 0)");
const PEAK_MODE = String(configPlugin.PeakMode || "dynamic");
const PEAK_COLOR_FIXED = String(configPlugin.PeakColorFixed || "rgb(251, 174, 38)");

// ====================================================================================
//  PATH DEFINITIONS FOR CLIENT FILES
// ====================================================================================
const MetricsMonitorClientFile = path.join(__dirname, "metricsmonitor.js");
const MetricsMonitorClientAnalyzerFile = path.join(
  __dirname,
  "js/metricsmonitor-analyzer.js"
);
const MetricsMonitorClientMetersFile = path.join(
  __dirname,
  "js/metricsmonitor-meters.js"
);
const MetricsMonitorClientAudioMeterFile = path.join(
  __dirname,
  "js/metricsmonitor-audiometer.js"
);
const MetricsMonitorClientHeaderFile = path.join(
  __dirname,
  "js/metricsmonitor-header.js"
);
const MetricsMonitorClientSignalMeterFile = path.join(
  __dirname,
  "js/metricsmonitor-signalmeter.js"
);
const MetricsMonitorClientSignalAnalyzerFile = path.join(
  __dirname,
  "js/metricsmonitor-signal-analyzer.js"
);

/**
 * HELPER: sequenceContainsId
 * Checks if a module ID is present in the sequence string/array.
 */
function sequenceContainsId(seq, id) {
  let arr;
  if (Array.isArray(seq)) {
    arr = seq;
  } else {
    arr = String(seq)
      .split(",")
      .map((s) => Number(s.trim()))
      .filter((n) => !Number.isNaN(n));
  }
  return arr.includes(id);
}

// Determine if the MPX processing should run
const ENABLE_MPX =
  sequenceContainsId(MODULE_SEQUENCE, 1) || // Analyzer Module
  sequenceContainsId(MODULE_SEQUENCE, 2) || // Meters Module
  sequenceContainsId(CANVAS_SEQUENCE, 2);   // MPX Canvas

const ENABLE_ANALYZER = ENABLE_MPX;

// ====================================================================================
//  DEPENDENCY MANAGEMENT
//  Ensures that necessary node modules are installed (e.g., fft-js).
// ====================================================================================

const RequiredModules = [
  "fft-js",
  "bit-twiddle",
  // If pffft.wasm is selected, we check it, otherwise we skip
  FFT_LIBRARY === "pffft.wasm" ? "@echogarden/pffft-wasm" : null,
].filter(Boolean);

function ensureRequiredModules() {
  RequiredModules.forEach((moduleName) => {
    const modulePath = path.join(__dirname, "./../../node_modules", moduleName);
    if (!fs.existsSync(modulePath)) {
      logInfo(`[MPX] Module "${moduleName}" is missing. Installing via npm...`);
      try {
        execSync(`npm install ${moduleName}`, { stdio: "inherit" });
        logInfo(`[MPX] Module "${moduleName}" installed successfully.`);
      } catch (error) {
        logError(`[MPX] Error installing module "${moduleName}":`, error);
      }
    }
  });
}

// ====================================================================================
//  SYSTEM PATCHING
//  These functions modify other server files to ensure compatibility.
// ====================================================================================

/**
 * PATCH 3LAS SERVER
 * Modifies '3las.server.js' to respect the 'sampleRate' from config instead of
 * being hardcoded to 48000 Hz (unless on Windows where 48k is forced).
 */
function patch3LAS() {
  try {
    const filePath = path.resolve(
      __dirname,
      "../../server/stream/3las.server.js"
    );
    let content = fs.readFileSync(filePath, "utf8");

    // We look for the standard 3LAS initialization line
    const oldBlockRegex = /const audioChannels[\s\S]*?48000\);/;

    const newBlock = `
const audioChannels = serverConfig.audio.audioChannels || 2;

// Default fallback
let sampleRate = Number(serverConfig.audio.sampleRate) || 48000;

// On Windows we still force 48000 Hz (3LAS limitation / compatibility)
if (process.platform === "win32") {
  sampleRate = 48000;
  logInfo("[Audio Stream] 3LAS on Windows detected → forcing sampleRate = 48000");
} else {
  logInfo("[Audio Stream] 3LAS using sampleRate from serverConfig.audio.sampleRate →", sampleRate);
}

const Server = new StreamServer(null, audioChannels, sampleRate);
    `.trim();

    if (oldBlockRegex.test(content)) {
      content = content.replace(oldBlockRegex, newBlock);
      fs.writeFileSync(filePath, content, "utf8");
      logInfo(
        "[MPX] 3LAS sampleRate block successfully patched. Please restart the webserver."
      );
    } else {
      // It might be already patched or different version
      logInfo(
        "[MPX] 3LAS old sampleRate block not found – no changes applied (possibly already patched)."
      );
    }
  } catch (err) {
    logError("[MPX] Failed to patch 3las.server.js:", err);
  }
}

/**
 * PATCH HELPERS.JS
 * Adds an exemption for Localhost (127.0.0.1) to bypass the Anti-Spam protection.
 * This is critical because the plugin communicates via internal WebSocket on localhost.
 */
const LOCALHOST_PATCH_MARKER = "// MM_LOCALHOST_SPAM_BYPASS:";

function patchHelpersForLocalhostBypass() {
  try {
    const helpersPath = path.join(__dirname, "./../../server/helpers.js");

    if (!fs.existsSync(helpersPath)) {
      logWarn(
        "[MPX] helpers.js not found, cannot patch antispamProtection()."
      );
      return;
    }

    let content = fs.readFileSync(helpersPath, "utf8");

    if (content.includes(LOCALHOST_PATCH_MARKER)) {
      // Already patched
      return;
    }

    // Locate the function
    const fnSignature =
      "function antispamProtection(message, clientIp, ws, userCommands, lastWarn, userCommandHistory, lengthCommands, endpointName) {";
    const fnIndex = content.indexOf(fnSignature);

    if (fnIndex === -1) {
      logWarn(
        "[MPX] antispamProtection() not found in helpers.js – skipping localhost patch."
      );
      return;
    }

    // Locate start of function body
    const commandLine = "const command = message.toString();";
    const cmdIndex = content.indexOf(commandLine, fnIndex);

    if (cmdIndex === -1) {
      logWarn(
        "[MPX] 'const command = message.toString();' not found in antispamProtection() – skipping localhost patch."
      );
      return;
    }

    const insertPos = cmdIndex + commandLine.length;

    const insertion = `
  ${LOCALHOST_PATCH_MARKER} allow internal server apps on localhost
  const isLocalhost =
    clientIp === "127.0.0.1" ||
    clientIp === "::1" ||
    clientIp === "::ffff:127.0.0.1" ||
    (clientIp && clientIp.replace(/^::ffff:/, '') === "127.0.0.1");

  if (isLocalhost) {
    // no spam/bot checks for local server applications
    return command;
  }`;

    content = content.slice(0, insertPos) + insertion + content.slice(insertPos);
    fs.writeFileSync(helpersPath, content, "utf8");

    logInfo(
      "[MPX] helpers.js patched: localhost exempt in antispamProtection(). Please restart the webserver!"
    );
  } catch (err) {
    logWarn(
      `[MPX] Failed to patch helpers.js for localhost exemption: ${err.message}`
    );
  }
}

// ====================================================================================
//  CLIENT-SIDE FILE UPDATES
//  Injects the current server configuration directly into the client .js files.
// ====================================================================================

/**
 * Normalize sequence to JSON string for injection
 */
function normalizeSequenceJS(seq) {
  if (Array.isArray(seq)) {
    return JSON.stringify(seq);
  }
  if (typeof seq === "string") {
    const items = seq
      .split(",")
      .map((s) => s.trim())
      .filter((s) => s.length > 0)
      .map(Number);
    return JSON.stringify(items);
  }
  return "[0, 1, 2, 3, 4]";
}

const MODULE_SEQUENCE_JS = normalizeSequenceJS(MODULE_SEQUENCE);
const CANVAS_SEQUENCE_JS = normalizeSequenceJS(CANVAS_SEQUENCE);

function updateSettings() {
  function buildHeaderBlock() {
    // This block is injected at the top of client files
    // NOTE: Sending renamed constants to the client to match new naming convention
    return (
      `const MODULE_SEQUENCE = ${MODULE_SEQUENCE_JS};    // Do not touch - this value is automatically updated via the config file\n` +
      `const CANVAS_SEQUENCE = ${CANVAS_SEQUENCE_JS};    // Do not touch - this value is automatically updated via the config file\n` +
      `const sampleRate = ${ANALYZER_SAMPLE_RATE};    // Do not touch - this value is automatically updated via the config file\n` +
      `const MeterInputCalibration = ${METER_INPUT_CALIBRATION_DB};    // Do not touch - this value is automatically updated via the config file\n` +
      `const MPXmode = "${MPX_MODE}";    // Do not touch - this value is automatically updated via the config file\n` +
      `const MPXStereoDecoder = "${MPX_STEREO_DECODER}";    // Do not touch - this value is automatically updated via the config file\n` +
      `const MPXInputCard = "${MPX_INPUT_CARD}";    // Do not touch - this value is automatically updated via the config file\n` +
      `const fftLibrary = "${FFT_LIBRARY}";    // Do not touch - this value is automatically updated via the config file\n` +
      `const fftSize = ${FFT_SIZE};    // Do not touch - this value is automatically updated via the config file\n` +
      `const SpectrumAttackLevel = ${SPECTRUM_ATTACK_LEVEL};    // Do not touch - this value is automatically updated via the config file\n` +
      `const SpectrumDecayLevel = ${SPECTRUM_DECAY_LEVEL};    // Do not touch - this value is automatically updated via the config file\n` +
      `const SpectrumSendInterval = ${SPECTRUM_SEND_INTERVAL};    // Do not touch - this value is automatically updated via the config file\n` +
      `const MeterPilotCalibration = ${METER_PILOT_CALIBRATION};    // Do not touch - this value is automatically updated via the config file\n` +
      `const MeterMPXCalibration = ${METER_MPX_CALIBRATION};    // Do not touch - this value is automatically updated via the config file\n` +
      `const MeterRDSCalibration = ${METER_RDS_CALIBRATION};    // Do not touch - this value is automatically updated via the config file\n` +
      `const SpectrumYOffset = ${SPECTRUM_Y_OFFSET};    // Do not touch - this value is automatically updated via the config file\n` +
      `const SpectrumYDynamics = ${SPECTRUM_Y_DYNAMICS};    // Do not touch - this value is automatically updated via the config file\n` +
      `const StereoBoost = ${STEREO_BOOST};    // Do not touch - this value is automatically updated via the config file\n` +
      `const AudioMeterBoost = ${AUDIO_METER_BOOST};    // Do not touch - this value is automatically updated via the config file\n` +
      `const LockVolumeSlider = ${LOCK_VOLUME_SLIDER};    // Do not touch - this value is automatically updated via the config file\n` +
      `const EnableSpectrumOnLoad = ${ENABLE_SPECTRUM_ON_LOAD};    // Do not touch - this value is automatically updated via the config file\n` +
      `const MeterColorSafe = ${METER_COLOR_SAFE};    // Do not touch - this value is automatically updated via the config file\n` +
      `const MeterColorWarning = ${METER_COLOR_WARNING};    // Do not touch - this value is automatically updated via the config file\n` +
      `const MeterColorDanger = ${METER_COLOR_DANGER};    // Do not touch - this value is automatically updated via the config file\n` +
      `const PeakMode = "${PEAK_MODE}";    // Do not touch - this value is automatically updated via the config file\n` +
      `const PeakColorFixed = "${PEAK_COLOR_FIXED}";    // Do not touch - this value is automatically updated via the config file\n`
    );
  }

  function removeOldConstants(code) {
    // Regex to remove existing constant definitions to prevent duplicates
    let out = code
      // Old names
      .replace(/^\s*const\s+minSendIntervalMs\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+CurveYOffset\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+CurveYDynamics\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+Curve-Y-Offset\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+Curve-Y-Dynamics\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+stereoBoost\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+eqBoost\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+EqBoost\s*=.*;[^\n]*\n?/gm, "")

      // Renamed names
      .replace(/^\s*const\s+MPXinputCalibration\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+MPXboost\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+MPXinputCalibrationDB\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+pilotCalibration\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+mpxCalibration\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+rdsCalibration\s*=.*;[^\n]*\n?/gm, "")

      // New names (to ensure clean update)
      .replace(/^\s*const\s+SpectrumSendInterval\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+SpectrumYOffset\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+SpectrumYDynamics\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+StereoBoost\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+AudioMeterBoost\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+MeterInputCalibration\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+MeterPilotCalibration\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+MeterMPXCalibration\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+MeterRDSCalibration\s*=.*;[^\n]*\n?/gm, "")

      // Other standard constants
      .replace(/^\s*const\s+MODULE_SEQUENCE\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+CANVAS_SEQUENCE\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+sampleRate\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+fftLibrary\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+fftSize\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+SpectrumAverageLevel\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+SpectrumAttackLevel\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+SpectrumDecayLevel\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+MPXmode\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+ExtStereoDecoder\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+MPXStereoDecoder\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+MPXInputCard\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+LockVolumeSlider\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+EnableSpectrumOnLoad\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+MeterColorSafe\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+MeterColorWarning\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+MeterColorDanger\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+PeakMode\s*=.*;[^\n]*\n?/gm, "")
      .replace(/^\s*const\s+PeakColorFixed\s*=.*;[^\n]*\n?/gm, "");

    out = out.replace(
      /^\s*\/\/\s*Do not touch - this value is automatically updated via the config file\s*$/gm,
      ""
    );

    return out;
  }

  function insertAfterIIFE(code) {
    const cleaned = removeOldConstants(code);
    // Find the start of the Immediately Invoked Function Expression
    const iifePattern = /(\(\s*\)\s*=>\s*\{)[ \t]*\n?/;

    if (!iifePattern.test(cleaned)) {
      return cleaned;
    }

    return cleaned.replace(
      iifePattern,
      (_, prefix) => `${prefix}\n${buildHeaderBlock()}`
    );
  }

  // ====================================================================================
  //  CLIENT PATCH: METERS DEBUG LOGGING + MPX/RDS HANDLING
  // ====================================================================================
  function patchMetersClient(code) {
    let out = code;

    // 1) Inject debug helpers once (search marker: mm_meters_debug)
    if (!out.includes("mm_meters_debug")) {
      const debugBlock = `
    // ============================
    // DEBUG LOGGING (CLIENT)
    // Enable:
    //   localStorage.setItem("mm_meters_debug","1");   // throttled
    //   localStorage.setItem("mm_meters_debug","all"); // EVERY WS message
    // Disable:
    //   localStorage.removeItem("mm_meters_debug");
    // Also supported: URL param ?mmMetersDebug=1
    // ============================
    const MM_METERS_DEBUG_MODE =
      (localStorage.getItem("mm_meters_debug") ||
        (typeof location !== "undefined" && /(?:\\?|&)mmMetersDebug=1\\b/.test(location.search) ? "1" : ""));

    const MM_METERS_DEBUG_ALL = (MM_METERS_DEBUG_MODE === "all");
    const MM_METERS_DEBUG_ON  = !!MM_METERS_DEBUG_MODE;

    let _mmMetersLastLogMs = 0;

    function mmMetersLog(tag, data, throttleMs = 500) {
      if (!MM_METERS_DEBUG_ON) return;
      if (!MM_METERS_DEBUG_ALL) {
        const now = Date.now();
        if ((now - _mmMetersLastLogMs) < throttleMs) return;
        _mmMetersLastLogMs = now;
      }
      try { console.log("[MM Meters] " + tag, data); } catch {}
    }
`;
      const reVals = new RegExp(
        "(let\\s+valMpx\\s*=\\s*0;\\s*\\n\\s*let\\s+valPilot\\s*=\\s*0;\\s*\\n\\s*let\\s+valRds\\s*=\\s*0;\\s*\\n)"
      );
      out = out.replace(reVals, "$1" + debugBlock + "\n");
    }

    // 2) Fix early return that kills MPX/RDS updates
    const reEarlyReturn = new RegExp(
      "^\\s*//\\s*Ignore spectrum array[^\\n]*\\n\\s*if\\s*\\(\\s*Array\\.isArray\\(message\\.value\\)\\s*\\)\\s*\\{\\s*return;\\s*\\}\\s*\\n",
      "m"
    );
    out = out.replace(
      reEarlyReturn,
      "        // NOTE: The server always sends 'value' as spectrum array alongside peak/pilotKHz/rdsKHz.\n" +
      "        // Do NOT return here. Only ignore spectrum-only packets that lack 'peak'.\n" +
      "        if (typeof message.peak !== \"number\" && Array.isArray(message.value)) { return; }\n\n"
    );

    // 3) Add detailed logging to handleMpxMessage (log complete MPX payload + extracted values)
    const reHandleHead = new RegExp(
      "(function\\s+handleMpxMessage\\s*\\(message\\)\\s*\\{\\s*\\n\\s*if\\s*\\(!message\\s*\\|\\|\\s*typeof\\s+message\\s*!==\\s*[\"']object[\"']\\)\\s*return;\\s*\\n)"
    );
    out = out.replace(
      reHandleHead,
      "$1" +
      "        try {\n" +
      "          const safe = Object.assign({}, message);\n" +
      "          if (Array.isArray(safe.value)) {\n" +
      "            safe.valueLen = safe.value.length;\n" +
      "            safe.valuePreview = safe.value.slice(0, 12);\n" +
      "            delete safe.value;\n" +
      "          }\n" +
      "          mmMetersLog(\"RX MPX (full)\", safe, 0);\n" +
      "          mmMetersLog(\"RX MPX VALUES\", {\n" +
      "            peak: message.peak,\n" +
      "            pilotKHz: message.pilotKHz,\n" +
      "            rdsKHz: message.rdsKHz,\n" +
      "            pilotRaw: message.pilot,\n" +
      "            rdsRaw: message.rds,\n" +
      "            noise: message.noise,\n" +
      "            snr: message.snr\n" +
      "          }, 0);\n" +
      "        } catch {}\n"
    );

    // 4) Expand WS onmessage logging (RAW + PARSED)
    const reWsCore = new RegExp(
      "const\\s+msg\\s*=\\s*JSON\\.parse\\(event\\.data\\);\\s*\\n\\s*//\\s*Safety:\\s*Don't process bare arrays[^\\n]*\\n\\s*if\\s*\\(Array\\.isArray\\(msg\\)\\)\\s*return;\\s*\\n\\s*\\n\\s*if\\s*\\(msg\\.type\\s*===\\s*[\"']MPX[\"']\\)\\s*handleMpxMessage\\(msg\\);",
      "m"
    );
    out = out.replace(
      reWsCore,
      "mmMetersLog(\"WS RAW\", event.data, 0);\n" +
      "          const msg = JSON.parse(event.data);\n" +
      "          if (Array.isArray(msg)) {\n" +
      "            mmMetersLog(\"WS PARSED (array)\", { len: msg.length, preview: msg.slice(0, 12) }, 0);\n" +
      "            return;\n" +
      "          }\n" +
      "          try {\n" +
      "            const safe = Object.assign({}, msg);\n" +
      "            if (Array.isArray(safe.value)) {\n" +
      "              safe.valueLen = safe.value.length;\n" +
      "              safe.valuePreview = safe.value.slice(0, 12);\n" +
      "              delete safe.value;\n" +
      "            }\n" +
      "            mmMetersLog(\"WS PARSED (object)\", safe, 0);\n" +
      "          } catch {}\n" +
      "          if (msg.type === \"MPX\") handleMpxMessage(msg);"
    );

    return out;
  }

  function updateClientFile(filePath, label, modifyFn) {
    try {
      const data = fs.readFileSync(filePath, "utf8");
      const updated = modifyFn(data);
      fs.writeFileSync(filePath, updated, "utf8");
    } catch (err) {
      logError(`[MPX] Error updating ${label}:`, err);
    }
  }

  // Update specific files
  updateClientFile(MetricsMonitorClientFile, "metricsmonitor.js", (code) => {
    let updated = code;
    const moduleSeqRegex = /^\s*const\s+MODULE_SEQUENCE\s*=.*;[^\n]*$/m;

    if (moduleSeqRegex.test(updated)) {
      updated = updated.replace(
        moduleSeqRegex,
        `const MODULE_SEQUENCE = ${MODULE_SEQUENCE_JS};    // Do not touch - this value is automatically updated via the config file`
      );
    } else {
      updated =
        `const MODULE_SEQUENCE = ${MODULE_SEQUENCE_JS};    // Do not touch - this value is automatically updated via the config file\n` +
        updated;
    }
    return insertAfterIIFE(updated);
  });

  updateClientFile(
    MetricsMonitorClientAnalyzerFile,
    "metricsmonitor-analyzer.js",
    insertAfterIIFE
  );
  updateClientFile(
    MetricsMonitorClientAudioMeterFile,
    "metricsmonitor-audiometer.js",
    insertAfterIIFE
  );
  updateClientFile(
    MetricsMonitorClientHeaderFile,
    "metricsmonitor-header.js",
    insertAfterIIFE
  );
  updateClientFile(
    MetricsMonitorClientMetersFile,
    "metricsmonitor-meters.js",
    (code) => patchMetersClient(insertAfterIIFE(code))
  );
  updateClientFile(
    MetricsMonitorClientSignalMeterFile,
    "metricsmonitor-signalmeter.js",
    insertAfterIIFE
  );
  updateClientFile(
    MetricsMonitorClientSignalAnalyzerFile,
    "metricsmonitor-signal-analyzer.js",
    insertAfterIIFE
  );
}

/**
 * DEPLOY CLIENT FILES
 * Copies the client-side files from the plugin directory to the Webserver's public `web` folder.
 */
function copyClientFiles() {
  if (process.platform === "win32") {
    logInfo("[MPX] Windows detected – skipping client file copy.");
    return;
  }

  const srcDir = __dirname;
  const destDir = path.join(__dirname, "../../web/js/plugins/MetricsMonitor");

  logInfo("[MPX] Updating client files in:", destDir);

  try {
    fs.mkdirSync(destDir, { recursive: true });
    fs.chmodSync(destDir, 0o775);
  } catch (e) {
    logError("[MPX] Failed to create destination directory:", e);
    return;
  }

  const folders = ["css", "js", "images"];

  folders.forEach((folder) => {
    const folderSrc = path.join(srcDir, folder);
    const folderDest = path.join(destDir, folder);

    if (!fs.existsSync(folderSrc)) return;

    fs.mkdirSync(folderDest, { recursive: true });
    try {
      fs.chmodSync(folderDest, 0o775);
    } catch {}

    const items = fs.readdirSync(folderSrc);
    items.forEach((item) => {
      const s = path.join(folderSrc, item);
      const d = path.join(folderDest, item);
      try {
        fs.copyFileSync(s, d);
        fs.chmodSync(d, 0o664);
        logInfo(`[MPX] Copied client file: ${d}`);
      } catch (err) {
        logError("[MPX] Error copying client file:", err);
      }
    });
  });

  const singleFiles = ["metricsmonitor.js"];
  singleFiles.forEach((file) => {
    const s = path.join(srcDir, file);
    const d = path.join(destDir, file);
    if (!fs.existsSync(s)) return;
    try {
      fs.copyFileSync(s, d);
      fs.chmodSync(d, 0o664);
      logInfo(`[MPX] Copied client root file: ${file}`);
    } catch (err) {
      logError("[MPX] Failed to copy client root file", file, err);
    }
  });
}

// Run update routines immediately
updateSettings();
copyClientFiles();

// ====================================================================================
//  MAIN SERVER LOGIC START
//  This is where the actual signal processing loop begins.
// ====================================================================================

if (!ENABLE_MPX) {
  logInfo(
    `[MPX] MODULE_SEQUENCE = ${MODULE_SEQUENCE} → ` +
    "MPX capture & server-side MPX processing are disabled."
  );
} else {
  // If enabled, proceed to setup
  ensureRequiredModules();

  let FFT = null;
  let pffftModule = null;
  let pffftSetup = null;
  let pffftInputPtr = null;
  let pffftOutputPtr = null;
  let pffftWorkPtr = null;
  let pffftInputHeap = null;
  let pffftOutputHeap = null;
  let fftReady = false;

  // Initialize FFT Library (WASM or JS fallback)
  if (FFT_LIBRARY === "pffft.wasm") {
    logInfo("[MPX] Initializing pffft.wasm library...");
    (async () => {
      try {
        const PFFFT = await import("@echogarden/pffft-wasm");
        pffftModule = await PFFFT.default();

        pffftSetup = pffftModule._pffft_new_setup(FFT_SIZE, 0);
        pffftInputPtr = pffftModule._pffft_aligned_malloc(FFT_SIZE * 4);
        pffftOutputPtr = pffftModule._pffft_aligned_malloc(FFT_SIZE * 4);
        pffftWorkPtr = pffftModule._pffft_aligned_malloc(FFT_SIZE * 4);

        pffftInputHeap = new Float32Array(
          pffftModule.HEAPF32.buffer,
          pffftInputPtr,
          FFT_SIZE
        );
        pffftOutputHeap = new Float32Array(
          pffftModule.HEAPF32.buffer,
          pffftOutputPtr,
          FFT_SIZE
        );

        fftReady = true;
        logInfo(
          `[MPX] pffft.wasm initialized successfully for FFT size ${FFT_SIZE}`
        );
      } catch (e) {
        logError("[MPX] Failed to initialize pffft.wasm:", e);
        FFT = require("fft-js").fft;
        fftReady = true;
        logInfo("[MPX] Fallback to 'fft-js' active.");
      }
    })();
  } else {
    FFT = require("fft-js").fft;
    fftReady = true;
    logInfo("[MPX] Using standard 'fft-js' library.");
  }

  // Apply patches
  patch3LAS();
  patchHelpersForLocalhostBypass();

  let SAMPLE_RATE = CONFIG_SAMPLE_RATE;
  const HOP_SIZE = FFT_SIZE / 2;
  const MAX_LATENCY_BLOCKS = 2;

  let SERVER_PORT = 8080;
  try {
    if (mainConfig?.webserver?.webserverPort) {
      SERVER_PORT = parseInt(mainConfig.webserver.webserverPort, 10);
      if (isNaN(SERVER_PORT)) SERVER_PORT = 8080;
    }
  } catch (e) {
    SERVER_PORT = 8080;
  }

  logInfo(`[MPX] Using webserver port from config.json → ${SERVER_PORT}`);
  logInfo(
    `[MPX] sampleRate from metricsmonitor.json → ${CONFIG_SAMPLE_RATE} Hz`
  );
  logInfo(`[MPX] FFT Library → ${FFT_LIBRARY}`);
  logInfo(`[MPX] FFT_SIZE from metricsmonitor.json → ${FFT_SIZE} points`);
  logInfo(`[MPX] Analyzer enabled? → ${ENABLE_ANALYZER}`);
  logInfo(
    `[MPX] SpectrumSendInterval from metricsmonitor.json → ${SPECTRUM_SEND_INTERVAL} ms`
  );
  logInfo(`[MPX] MPXmode from metricsmonitor.json → ${MPX_MODE}`);
  logInfo(
    `[MPX] MPXStereoDecoder from metricsmonitor.json → ${MPX_STEREO_DECODER}`
  );
  
  // Separate Calibration Logs
  logInfo(`[MPX] MeterInputCalibration (Meters) → ${METER_INPUT_CALIBRATION_DB} dB (Factor: ${METER_GAIN_FACTOR.toFixed(3)})`);
  logInfo(`[MPX] SpectrumInputCalibration (Spectrum) → ${SPECTRUM_INPUT_CALIBRATION_DB} dB (Factor: ${SPECTRUM_GAIN_FACTOR.toFixed(3)})`);

  if (MPX_INPUT_CARD !== "") {
    logInfo(`[MPX] MPXInputCard from metricsmonitor.json → "${MPX_INPUT_CARD}"`);
  }

  // ====================================================================================
  //  BINARY SELECTION LOGIC
  //  Selects the correct MPXCapture binary based on OS/Arch.
  // ====================================================================================
  const osPlatform = process.platform;
  const osArch = process.arch;

  let runtimeFolder = null;
  let binaryName = null;

  if (osPlatform === "win32") {
    const archEnv = process.env.PROCESSOR_ARCHITECTURE || "";
    const archWow = process.env.PROCESSOR_ARCHITEW6432 || "";
    const is64BitOS =
      archEnv.toUpperCase() === "AMD64" || archWow.toUpperCase() === "AMD64";

    runtimeFolder = is64BitOS ? "win-x64" : "win-x86";
    binaryName = "MPXCapture.exe";
  } else if (osPlatform === "linux") {
    if (osArch === "arm" || osArch === "armhf") {
      runtimeFolder = "linux-arm";
    } else if (osArch === "arm64") {
      runtimeFolder = "linux-arm64";
    } else {
      runtimeFolder = "linux-x64";
    }
    binaryName = "MPXCapture";
  } else if (osPlatform === "darwin") {
    runtimeFolder = osArch === "arm64" ? "osx-arm64" : "osx-x64";
    binaryName = "MPXCapture";
  } else {
    logError(
      `[MPX] Unsupported platform ${osPlatform}/${osArch} – MPXCapture will not be started.`
    );
  }

  let MPX_EXE_PATH = null;

  if (!runtimeFolder || !binaryName) {
    logWarn(
      "[MPX] No runtimeFolder/binaryName detected – MPXCapture disabled."
    );
  } else if (
    osPlatform === "win32" &&
    CONFIG_SAMPLE_RATE === 48000 &&
    MPX_INPUT_CARD === ""
  ) {
    logWarn(
      "[MPX] CONFIG_SAMPLE_RATE = 48000 on Windows (and no MPXInputCard) → using 3LAS, MPXCapture disabled."
    );
  } else {
    MPX_EXE_PATH = path.join(__dirname, "bin", runtimeFolder, binaryName);
    MPX_EXE_PATH = MPX_EXE_PATH.replace(/^['\"]+|['\"]+$/g, "");
    logInfo(
      `[MPX] Using MPXCapture binary for ${osPlatform}/${osArch} → ${runtimeFolder}/${binaryName}`
    );
  }

  const BIN_STEP = 2;
  const MAX_WS_BACKLOG_BYTES = 256 * 1024;

  logInfo(
    "[MPX] MPX server started (Fast & Smooth v2.1, Peak/Pilot/RDS Time Domain)."
  );

  // ====================================================================================
  //  WEBSOCKET SERVER
  //  Handles data distribution to clients.
  // ====================================================================================
  let dataPluginsWs = null;
  let reconnectTimer = null;
  let backpressureHits = 0;
  const MAX_BACKPRESSURE_HITS = 200;

  function connectDataPluginsWs() {
    const url = `ws://127.0.0.1:${SERVER_PORT}/data_plugins`;

    if (
      dataPluginsWs &&
      (dataPluginsWs.readyState === WebSocket.OPEN ||
        dataPluginsWs.readyState === WebSocket.CONNECTING)
    ) {
      return;
    }

    logInfo("[MPX] Connecting to /data_plugins:", url);

    dataPluginsWs = new WebSocket(url);
    backpressureHits = 0;

    dataPluginsWs.on("open", () => {
      logInfo("[MPX] Connected to /data_plugins WebSocket.");
      backpressureHits = 0;
    });

    dataPluginsWs.on("close", () => {
      logInfo("[MPX] /data_plugins WebSocket closed – retrying in 5 seconds.");
      dataPluginsWs = null;

      if (!reconnectTimer) {
        reconnectTimer = setTimeout(() => {
          reconnectTimer = null;
          connectDataPluginsWs();
        }, 5000);
      }
    });

    dataPluginsWs.on("error", (err) => {
      logError("[MPX] /data_plugins WebSocket error:", err);
    });

    dataPluginsWs.on("message", () => {});
  }

  connectDataPluginsWs();

  // ====================================================================================
  //  SIGNAL PROCESSING BUFFERS & VARIABLES
  // ====================================================================================
  let use3LasPcmFormat = false;
  let sampleBuffer = [];

  const GOERTZEL_WINDOW_SIZE = 4096;
  let goertzelAccumulator = [];
  let goertzelWindow = null;
  let debugLogCounter = 0;

  let currentMaxPeak = 0;
  let currentPilotPeak = 0;
  let currentRdsPeak = 0;
  let currentNoiseFloor = 0;

  // Tracking Min/Max for Debugging
  let statPilotMin = 999.0;
  let statPilotMax = 0.0;
  let statRdsMin = 999.0;
  let statRdsMax = 0.0;
  let statMpxPeakMin = 999.0;
  let statMpxPeakMax = 0.0;

  const GOERTZEL_PILOT_FREQ = 19000;
  const GOERTZEL_RDS_FREQ_1 = 56000;
  const GOERTZEL_RDS_FREQ_2 = 57000;
  const GOERTZEL_RDS_FREQ_3 = 58000;
  const GOERTZEL_NOISE_FREQ = 25000;

  // Deviation Statistics (Missing in previous version)
  let devSamplesTotal = 0;
  let devSamplesExceed = 0;
  const MODPOWER_WINDOW_S = 60;
  let mpBlockQueue = [];
  let mpIntegral60 = 0;
  let mpDur60 = 0;

  let fftBlock = null;
  let windowHann = null;

  if (ENABLE_ANALYZER) {
    fftBlock = new Float32Array(FFT_SIZE);
    windowHann = new Float32Array(FFT_SIZE);
    for (let i = 0; i < FFT_SIZE; i++) {
      windowHann[i] = 0.5 * (1 - Math.cos((2 * Math.PI * i) / (FFT_SIZE - 1)));
    }
  }

  let latestMpxFrame = null;

  /**
   * Initialize the Hann Window for Goertzel Analysis
   * This improves frequency separation and reduces spectral leakage.
   */
  function initGoertzelWindow() {
    goertzelWindow = new Float32Array(GOERTZEL_WINDOW_SIZE);
    for (let i = 0; i < GOERTZEL_WINDOW_SIZE; i++) {
      goertzelWindow[i] =
        0.5 * (1 - Math.cos((2 * Math.PI * i) / (GOERTZEL_WINDOW_SIZE - 1)));
    }
    logInfo(
      `[MPX] Precision Analyzer initialized. Window Size: ${GOERTZEL_WINDOW_SIZE}`
    );
  }

  // ====================================================================================
  //  GOERTZEL ALGORITHM IMPLEMENTATION
  //  Calculates the magnitude of a specific frequency component in the signal.
  // ====================================================================================
  function calculateGoertzelWindowed(samples, targetFreq, sampleRate) {
    if (sampleRate < 1000) return 0;

    const omega = (2.0 * Math.PI * targetFreq) / sampleRate;
    const cosine = Math.cos(omega);
    const sine = Math.sin(omega);
    const coeff = 2.0 * cosine;

    let q0 = 0,
      q1 = 0,
      q2 = 0;

    for (let i = 0; i < samples.length; i++) {
      const windowedSample = samples[i] * goertzelWindow[i];
      q0 = coeff * q1 - q2 + windowedSample;
      q2 = q1;
      q1 = q0;
    }

    const real = q1 - q2 * cosine;
    const imag = q2 * sine;

    return (Math.sqrt(real * real + imag * imag) * 2.0) / samples.length * 2.0;
  }

 // ====================================================================================
  //  PCM DATA HANDLER (V10.23: RAW PEAK DETECTOR)
  //  Fix: Removed RMS Limiter which was clamping valid MPX peaks on compressed stations.
  // ====================================================================================
  
  // Ensure global variable exists
  if (typeof global.currentAudioLevel === 'undefined') global.currentAudioLevel = 0;

  function handlePcmChunk(chunk) {
    if (!chunk.length) return;
    if (!goertzelWindow) initGoertzelWindow();

    let meterBuffer = [];
    if (use3LasPcmFormat) {
        const intData = new Int16Array(chunk.buffer, chunk.byteOffset, chunk.byteLength / 2);
        for (let i = 0; i < intData.length; i += 2) {
            const v = ((intData[i]/32768.0) + ((intData[i+1]??intData[i])/32768.0))*0.5;
            if (ENABLE_ANALYZER) sampleBuffer.push(v * SPECTRUM_GAIN_FACTOR);
            meterBuffer.push(v * METER_GAIN_FACTOR);
        }
    } else {
        const floatData = new Float32Array(chunk.buffer, chunk.byteOffset, chunk.byteLength / 4);
        for (let i = 0; i < floatData.length; i += 2) {
            const v = (floatData[i] + (floatData[i+1]??floatData[i]))*0.5;
            if (ENABLE_ANALYZER) sampleBuffer.push(v * SPECTRUM_GAIN_FACTOR);
            meterBuffer.push(v * METER_GAIN_FACTOR);
        }
    }

    // Time Domain Peak (Raw & Unfiltered)
    let chunkMax = 0;
    for (let i = 0; i < meterBuffer.length; i++) {
      let val = Math.abs(meterBuffer[i]);
      if (val > chunkMax) chunkMax = val;
      if (goertzelWindow) goertzelAccumulator.push(meterBuffer[i]);
    }
    
    // Smooth decay for the max peak to catch fast transients
    if (chunkMax >= currentMaxPeak) {
        currentMaxPeak = chunkMax;
    } else {
        // Fast recovery to allow modulation to "dance"
        currentMaxPeak = currentMaxPeak * 0.95 + chunkMax * 0.05;
    }

    // Goertzel Block Processing
    if (goertzelAccumulator.length > GOERTZEL_WINDOW_SIZE * 4) goertzelAccumulator = goertzelAccumulator.slice(-GOERTZEL_WINDOW_SIZE);
    
    const processingRate = (SAMPLE_RATE && SAMPLE_RATE > 0) ? SAMPLE_RATE : 48000;

    while (goertzelAccumulator.length >= GOERTZEL_WINDOW_SIZE) {
        const chunk = goertzelAccumulator.slice(0, GOERTZEL_WINDOW_SIZE);
        goertzelAccumulator = goertzelAccumulator.slice(Math.floor(GOERTZEL_WINDOW_SIZE*0.5));
        
        // 1. Pilot (19k)
        const pilot = calculateGoertzelWindowed(chunk, 19000, processingRate);
        
        // 2. Stereo Carrier (38k)
        const stereo38k = calculateGoertzelWindowed(chunk, 38000, processingRate);

        // 3. RDS Measurement (Sidebands)
        const rdsLower = calculateGoertzelWindowed(chunk, 56000, processingRate);
        const rdsUpper = calculateGoertzelWindowed(chunk, 58000, processingRate);
        let rawRdsEnergy = (rdsLower + rdsUpper) / 2.0;

        // 4. Noise Profile
        const n1 = calculateGoertzelWindowed(chunk, 17000, processingRate);
        const n2 = calculateGoertzelWindowed(chunk, 21000, processingRate); 
        const n3 = calculateGoertzelWindowed(chunk, 54500, processingRate); 
        const n4 = calculateGoertzelWindowed(chunk, 65000, processingRate);
        const n5 = calculateGoertzelWindowed(chunk, 85000, processingRate); 
        const avgNoise = (n1 + n2 + n3 + n4 + n5) / 5.0;

        // 5. Audio Energy Check
        const audio400 = calculateGoertzelWindowed(chunk, 400, processingRate);
        const audio1k  = calculateGoertzelWindowed(chunk, 1000, processingRate);
        const audio3k  = calculateGoertzelWindowed(chunk, 3000, processingRate);
        const rawAudio = (audio400 + audio1k + audio3k) / 3.0;

        // Update Globals
        currentNoiseFloor = currentNoiseFloor * 0.90 + avgNoise * 0.10;
        currentPilotPeak = currentPilotPeak * 0.85 + pilot * 0.15;
        currentRdsPeak = currentRdsPeak * 0.98 + rawRdsEnergy * 0.02; // Slow RDS Avg
        
        if (typeof global.currentStereoCarrier === 'undefined') global.currentStereoCarrier = 0;
        global.currentStereoCarrier = global.currentStereoCarrier * 0.9 + stereo38k * 0.1;

        if (typeof global.currentAudioLevel === 'undefined') global.currentAudioLevel = 0;
        global.currentAudioLevel = global.currentAudioLevel * 0.9 + rawAudio * 0.1;
    }

    // FFT Visualization
    if (ENABLE_ANALYZER && fftReady) {
        if (sampleBuffer.length > MAX_LATENCY_BLOCKS * FFT_SIZE) sampleBuffer.splice(0, sampleBuffer.length - MAX_LATENCY_BLOCKS * FFT_SIZE);
        if (sampleBuffer.length >= FFT_SIZE) {
            const start = sampleBuffer.length - FFT_SIZE;
            for(let i=0; i<FFT_SIZE; i++) fftBlock[i] = sampleBuffer[start+i] * windowHann[i];
            const keep = Math.max(0, sampleBuffer.length - HOP_SIZE);
            if (keep > 0) sampleBuffer.splice(0, keep); else sampleBuffer.length = 0;

            let mags = new Float32Array(FFT_SIZE/2);
            if (pffftModule) {
                pffftInputHeap.set(fftBlock);
                pffftModule._pffft_transform_ordered(pffftSetup, pffftInputPtr, pffftOutputPtr, pffftWorkPtr, 0);
                for(let i=0; i<FFT_SIZE/2; i++) {
                    const re = pffftOutputHeap[2*i], im = pffftOutputHeap[2*i+1];
                    mags[i] = (Math.sqrt(re*re + im*im)/(FFT_SIZE/2)) * (i>0?10:1);
                }
            } else {
                 const phasors = FFT(fftBlock);
                 for(let i=0; i<FFT_SIZE/2; i++) mags[i] = (Math.sqrt(phasors[i][0]**2 + phasors[i][1]**2)/(FFT_SIZE/2)) * (i>0?10:1);
            }
            const mpx = [];
            for (let i = 0; i < FFT_SIZE/2; i+=2) {
                if ((i*processingRate/FFT_SIZE) > 100000) break;
                mpx.push(Math.round(mags[i]*100000)/100000);
            }
            if (mpx.length) latestMpxFrame = mpx;
        }
    }
  }
  
  // ====================================================================================
  //  AUDIO INPUT INITIALIZATION
  //  Selects between 3LAS stream or MPXCapture binary.
  // ====================================================================================
  let rec = null;
  const explicitCard =
    MPX_INPUT_CARD !== "" && MPX_INPUT_CARD.toLowerCase() !== "off";

  const USE_3LAS = !explicitCard && CONFIG_SAMPLE_RATE === 48000;

  if (USE_3LAS) {
    try {
      const audioStream = require("./../../server/stream/3las.server");

      if (!audioStream || !audioStream.waitUntilReady) {
        logWarn(
          "[MPX] 3LAS server not available – MPX spectrum capture disabled."
        );
      } else {
        audioStream.waitUntilReady
          .then(() => {
            const s = audioStream.Server;
            if (!s || !s.StdIn) {
              logError(
                "[MPX] 3LAS Server has no StdIn stream – MPX spectrum capture disabled."
              );
              return;
            }

            if (typeof s.SampleRate === "number" && s.SampleRate > 0) {
              SAMPLE_RATE = s.SampleRate;
            } else {
              SAMPLE_RATE = CONFIG_SAMPLE_RATE || 48000;
              logWarn(
                `[MPX] 3LAS sampleRate unknown – assuming ${SAMPLE_RATE} Hz for MPX spectrum.`
              );
            }

            use3LasPcmFormat = true;

            logInfo(
              `[MPX] Subscribing to 3LAS StdIn PCM stream (${osPlatform}) @ ${SAMPLE_RATE} Hz`
            );

            s.StdIn.on("data", (buffer) => {
              handlePcmChunk(buffer);
            });
          })
          .catch((err) => {
            logError("[MPX] Error while waiting for 3LAS audio stream:", err);
          });
      }
    } catch (e) {
      logError(
        "[MPX] Failed to require 3las.server – MPX spectrum capture disabled:",
        e
      );
    }
  } else if (!MPX_EXE_PATH) {
    logWarn(
      "[MPX] MPXCapture path not resolved or platform unsupported – not starting MPXCapture."
    );
  } else if (!fs.existsSync(MPX_EXE_PATH)) {
    logError("[MPX] MPXCapture binary not found at path:", MPX_EXE_PATH);
  } else {
    use3LasPcmFormat = false;

    if (osPlatform !== "win32") {
      try {
        fs.chmodSync(MPX_EXE_PATH, 0o755);
      } catch (err) {
        logWarn(
          `[MPX] Failed to set execution permissions for ${MPX_EXE_PATH}:`,
          err
        );
      }
    }

    const args = [String(SAMPLE_RATE)];

    if (explicitCard) {
      logInfo(
        `[MPX] Starting MPXCapture with specific device: "${MPX_INPUT_CARD}"`
      );
      args.push(MPX_INPUT_CARD);
    } else {
      logInfo(
        `[MPX] Starting MPXCapture (${osPlatform}/${osArch}) with SAMPLE_RATE = ${SAMPLE_RATE} Hz (Default Device)`
      );
    }

    rec = spawn(MPX_EXE_PATH, args);

    rec.stderr.on("data", (d) => {
      const text = d.toString().trim();
      if (text.length > 0) {
        logInfo("[MPX-EXE]", text);
      }
    });

    rec.stdout.on("data", handlePcmChunk);

    rec.on("close", (code, signal) => {
      logInfo(
        "[MPX] MPXCapture exited with code:",
        code,
        "signal:",
        signal || "none"
      );
      if (explicitCard && code !== 0) {
        logWarn(
          `[MPX] MPXCapture exited prematurely. Verify if device "${MPX_INPUT_CARD}" exists and is not in use.`
        );
      }
    });
  }

  // ====================================================================================
  //  MAIN BROADCAST LOOP (CALIBRATED V10.7 - NOISE GATE & PILOT TRIGGER)
  //  Fix: MPX Noise Gate & Simplified Stereo Detection (Pilot > 6kHz = Stereo)
  // ====================================================================================
  
  // GLOBAL SCALING CONSTANTS (Matched to V9.1 Calibration)
  const PILOT_SCALE_KHZ_PER_AMP = 1060.0;
  const RDS_SCALE_KHZ_PER_AMP = 14500.0; 
  const DEV_SCALE_KHZ_PER_AMP = 525.0;
  
  // STATS
  let statMpxMin = 999;
  let statMpxMax = 0;

  // OUTPUT VALUES
  let out_mpx = 0;
  let out_pilot = 0;
  let out_rds = 0;

  // --- FILTER STATE INITIALIZATION ---
  if (typeof global.mpxPeakState === 'undefined') global.mpxPeakState = 0;
  if (typeof global.rdsSmoother === 'undefined') global.rdsSmoother = 0;
  if (typeof global.pilotFastAvg === 'undefined') global.pilotFastAvg = 0;
  if (typeof global.isStereoLocked === 'undefined') global.isStereoLocked = false;
  if (typeof global.signalHoldTimer === 'undefined') global.signalHoldTimer = 0;
  if (typeof global.mpxDisplayPeak === 'undefined') global.mpxDisplayPeak = 0;
  if (typeof global.rdsStableValue === 'undefined') global.rdsStableValue = 0;
  if (typeof global.logThrottle === 'undefined') global.logThrottle = 0;

  // Ensure global states exist
  if (typeof global.modHistory === 'undefined') global.modHistory = [];

  setInterval(() => {
    if (!dataPluginsWs || dataPluginsWs.readyState !== WebSocket.OPEN) return;
    if (dataPluginsWs.bufferedAmount > 262144) {
        if (++backpressureHits >= 50) { try { dataPluginsWs.terminate(); } catch {} dataPluginsWs = null; }
        return;
    }
    backpressureHits = 0;

    const PILOT_SCALE = 1060.0;
    const RDS_SCALE = 14500.0;
    const MPX_SCALE = 560.0; 

    if (typeof global.currentStereoCarrier === 'undefined') global.currentStereoCarrier = 0;

    let rP = currentPilotPeak;
    let rR = currentRdsPeak;
    let rM = currentMaxPeak;
    let rN = currentNoiseFloor || 0.000001;

    let snr = rP / rN;

    // --- STEP 2: DYNAMICS ---
    global.modHistory.push(rM);
    if (global.modHistory.length > 10) global.modHistory.shift(); 
    let totalDelta = 0;
    for (let i = 1; i < global.modHistory.length; i++) {
        totalDelta += Math.abs(global.modHistory[i] - global.modHistory[i-1]);
    }
    let isModulating = (totalDelta > 0.2) || (rN > 0.1);

    // --- STEP 3: SQUELCH ---
    // Basic validity check for signal presence
    let isPilotValid = (snr > 2.0 && rP > 0.0006); 
    if (!isPilotValid || rN > 0.35) { 
        rM=0; rP=0; rR=0;
        global.pilotFastAvg = 0; global.isStereoLocked = false; 
        global.signalHoldTimer = 0; global.rdsStableValue = 0;
        global.modHistory = []; 
    }

    if (isPilotValid) global.signalHoldTimer = 50; 
    else if (global.signalHoldTimer > 0) global.signalHoldTimer--;

    // --- STEP 4: CALCULATION ---
    let isPhysicallyMono = true; // Default to Mono

    if (global.signalHoldTimer > 0) {
        
        let cP = rP > rN ? rP - rN : rP;
        global.pilotFastAvg = (global.pilotFastAvg * 0.9) + (cP * 0.1);

        // Calculate Pilot Output FIRST to use it for decision
        out_pilot = (global.pilotFastAvg * PILOT_SCALE) + METER_PILOT_CALIBRATION;
        if (out_pilot < 0) out_pilot = 0;

        // USER REQUEST LOGIC: 
        // Stereo ONLY if Pilot > 6.0 kHz. Otherwise Mono.
        if (out_pilot > 5.5) {
            isPhysicallyMono = false;
            global.isStereoLocked = true;
        } else {
            isPhysicallyMono = true;
            global.isStereoLocked = false;
        }

        // RDS Output
        let cleanRds = 0;
        // Simple RDS check: Must be stronger than noise floor
        let rdsRelToNoise = (rR > (rN * 2.0)); 
        
        if (rdsRelToNoise && !isPhysicallyMono) {
             cleanRds = (rR > rN) ? (rR - rN) : rR;
        } else if (isPhysicallyMono) {
             cleanRds = 0; // Force RDS off in Mono
        } else {
             if (rR > (rN * 1.3)) cleanRds = rR - rN;
        }

        let rawKHz = (cleanRds * RDS_SCALE) + METER_RDS_CALIBRATION;
        if (rawKHz < 0) rawKHz = 0;

        let sm = (rawKHz > 1.5 && rawKHz < 4.0) ? ((global.rdsStableValue > 1.0) ? 0.02 : 0.05) : 0.2;
        global.rdsStableValue = (global.rdsStableValue * (1-sm)) + (rawKHz * sm);
        out_rds = global.rdsStableValue;
        if (out_rds > 12) out_rds = 12;

        // --- MPX PROCESSING ---
        // rM is Time-Domain Peak. Subtract Noise based on Mode.
        let cleanMpx = 0;
        
        if (isPhysicallyMono) {
             // MONO: Aggressive cleaning.
             // We assume everything above 15kHz is noise.
             let noiseSub = rN * 18.0; 
             
             // Clamp outlier peaks in Mono noise
             if (rM > (rN * 25.0)) rM = rN * 25.0; 
             
             cleanMpx = rM - noiseSub;
        } else {
             // STEREO: Standard noise floor removal.
             let noiseSubFactor = (snr < 20.0) ? 6.0 : 1.5; 
             cleanMpx = rM - (rN * noiseSubFactor);
        }
        
        if (cleanMpx < 0) cleanMpx = 0;
        let carrierSum = global.pilotFastAvg + (global.rdsStableValue / RDS_SCALE); 
        if (cleanMpx < carrierSum) cleanMpx = carrierSum;

        // Ballistics
        let lim = cleanMpx > 0.15 ? 0.15 + (cleanMpx - 0.15) * 0.1 : cleanMpx;
        
        let attack = (global.mpxPeakState < 0.02) ? 0.5 : 0.15;
        if (isPhysicallyMono) attack = 0.05; // Slower attack in Mono

        if (lim > global.mpxPeakState) {
            global.mpxPeakState = (global.mpxPeakState * (1 - attack)) + (lim * attack);
        } else global.mpxPeakState -= 0.0005;

        // Final Scaling
        const CORR = global.isStereoLocked ? 0.96 : 1.0;
        let calcMpx = (global.mpxPeakState * MPX_SCALE) + METER_MPX_CALIBRATION;
        calcMpx *= CORR;

        // --- CLAMPS & SANITY CHECKS ---
        if (isPhysicallyMono && !isModulating && calcMpx > 20.0) {
             calcMpx = out_pilot + 2.0;
        }
        
        // Safety clamp for Mono signals detecting false peaks
        if (isPhysicallyMono && calcMpx > 75.0) {
             calcMpx = 15.0 + out_pilot; 
        }

        let floor = out_pilot + out_rds;
        if (calcMpx < floor) calcMpx = floor;

        if (calcMpx >= global.mpxDisplayPeak) global.mpxDisplayPeak = calcMpx;
        else global.mpxDisplayPeak = (global.mpxDisplayPeak * 0.92) + (calcMpx * 0.08);

        if (global.mpxDisplayPeak < floor) global.mpxDisplayPeak = floor;
        out_mpx = global.mpxDisplayPeak;

    } else {
        out_mpx = 0; out_pilot = 0; out_rds = 0;
        global.isStereoLocked = false;
        global.mpxPeakState = 0; global.mpxDisplayPeak = 0; global.rdsStableValue = 0;
    }

    if (ENABLE_EXTENDED_LOGGING && global.signalHoldTimer > 0) {
        global.logThrottle++;
        if (global.logThrottle >= 33) { 
             global.logThrottle = 0;
             logInfo(`[DEBUG] Pilot:${out_pilot.toFixed(2)} | RDS:${out_rds.toFixed(2)} | MPX:${out_mpx.toFixed(2)} | Noise:${(rN*1000).toFixed(3)} | Mono:${isPhysicallyMono}`);
        }
    }

    const payload = JSON.stringify({
      type: "MPX", value: latestMpxFrame||[], peak: out_mpx, pilotKHz: out_pilot, rdsKHz: out_rds,
      pilot: rP, rds: rR, noise: rN, snr: (rN > 0.000001) ? (rP / rN) : 0
    });
    
    currentMaxPeak = 0; 
    dataPluginsWs.send(payload, ()=>{});
    
  }, SPECTRUM_SEND_INTERVAL);
  
  // Cleanup Handlers
  if (FFT_LIBRARY === "pffft.wasm") {
    process.on("exit", () => {
      if (pffftModule && pffftSetup) {
        try {
          pffftModule._pffft_destroy_setup(pffftSetup);
          pffftModule._pffft_aligned_free(pffftInputPtr);
          pffftModule._pffft_aligned_free(pffftOutputPtr);
          pffftModule._pffft_aligned_free(pffftWorkPtr);
        } catch (e) {}
      }
    });
  }
}